<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\MetalType\BulkDestroyMetalType;
use App\Http\Requests\Admin\MetalType\DestroyMetalType;
use App\Http\Requests\Admin\MetalType\IndexMetalType;
use App\Http\Requests\Admin\MetalType\StoreMetalType;
use App\Http\Requests\Admin\MetalType\UpdateMetalType;
use App\Models\MetalType;
use Brackets\AdminListing\Facades\AdminListing;
use Exception;
use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Contracts\View\Factory;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Response;
use Illuminate\Routing\Redirector;
use Illuminate\Support\Facades\DB;
use Illuminate\View\View;

class MetalTypeController extends Controller
{

    /**
     * Display a listing of the resource.
     *
     * @param IndexMetalType $request
     * @return array|Factory|View
     */
    public function index(IndexMetalType $request)
    {
        // create and AdminListing instance for a specific model and
        $data = AdminListing::create(MetalType::class)->processRequestAndGet(
            // pass the request with params
            $request,

            // set columns to query
            ['id', 'metal_type', 'status'],

            // set columns to searchIn
            ['id', 'metal_type']
        );

        if ($request->ajax()) {
            if ($request->has('bulk')) {
                return [
                    'bulkItems' => $data->pluck('id')
                ];
            }
            return ['data' => $data];
        }

        return view('admin.metal-type.index', ['data' => $data]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @throws AuthorizationException
     * @return Factory|View
     */
    public function create()
    {
        $this->authorize('admin.metal-type.create');

        return view('admin.metal-type.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreMetalType $request
     * @return array|RedirectResponse|Redirector
     */
    public function store(StoreMetalType $request)
    {
        // Sanitize input
        $sanitized = $request->getSanitized();

        // Store the MetalType
        $metalType = MetalType::create($sanitized);

        if ($request->ajax()) {
            return ['redirect' => url('admin/metal-types'), 'message' => trans('brackets/admin-ui::admin.operation.succeeded')];
        }

        return redirect('admin/metal-types');
    }

    /**
     * Display the specified resource.
     *
     * @param MetalType $metalType
     * @throws AuthorizationException
     * @return void
     */
    public function show(MetalType $metalType)
    {
        $this->authorize('admin.metal-type.show', $metalType);

        // TODO your code goes here
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param MetalType $metalType
     * @throws AuthorizationException
     * @return Factory|View
     */
    public function edit(MetalType $metalType)
    {
        $this->authorize('admin.metal-type.edit', $metalType);


        return view('admin.metal-type.edit', [
            'metalType' => $metalType,
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param UpdateMetalType $request
     * @param MetalType $metalType
     * @return array|RedirectResponse|Redirector
     */
    public function update(UpdateMetalType $request, MetalType $metalType)
    {
        // Sanitize input
        $sanitized = $request->getSanitized();

        // Update changed values MetalType
        $metalType->update($sanitized);

        if ($request->ajax()) {
            return [
                'redirect' => url('admin/metal-types'),
                'message' => trans('brackets/admin-ui::admin.operation.succeeded'),
            ];
        }

        return redirect('admin/metal-types');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param DestroyMetalType $request
     * @param MetalType $metalType
     * @throws Exception
     * @return ResponseFactory|RedirectResponse|Response
     */
    public function destroy(DestroyMetalType $request, MetalType $metalType)
    {
        $metalType->delete();

        if ($request->ajax()) {
            return response(['message' => trans('brackets/admin-ui::admin.operation.succeeded')]);
        }

        return redirect()->back();
    }

    /**
     * Remove the specified resources from storage.
     *
     * @param BulkDestroyMetalType $request
     * @throws Exception
     * @return Response|bool
     */
    public function bulkDestroy(BulkDestroyMetalType $request) : Response
    {
        DB::transaction(static function () use ($request) {
            collect($request->data['ids'])
                ->chunk(1000)
                ->each(static function ($bulkChunk) {
                    MetalType::whereIn('id', $bulkChunk)->delete();

                    // TODO your code goes here
                });
        });

        return response(['message' => trans('brackets/admin-ui::admin.operation.succeeded')]);
    }
}
