<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\Closure\BulkDestroyClosure;
use App\Http\Requests\Admin\Closure\DestroyClosure;
use App\Http\Requests\Admin\Closure\IndexClosure;
use App\Http\Requests\Admin\Closure\StoreClosure;
use App\Http\Requests\Admin\Closure\UpdateClosure;
use App\Models\Closure;
use Brackets\AdminListing\Facades\AdminListing;
use Exception;
use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Contracts\View\Factory;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Response;
use Illuminate\Routing\Redirector;
use Illuminate\Support\Facades\DB;
use Illuminate\View\View;

class ClosureController extends Controller
{

    /**
     * Display a listing of the resource.
     *
     * @param IndexClosure $request
     * @return array|Factory|View
     */
    public function index(IndexClosure $request)
    {
        // create and AdminListing instance for a specific model and
        $data = AdminListing::create(Closure::class)->processRequestAndGet(
            // pass the request with params
            $request,

            // set columns to query
            ['id', 'closure', 'status'],

            // set columns to searchIn
            ['id', 'closure']
        );

        if ($request->ajax()) {
            if ($request->has('bulk')) {
                return [
                    'bulkItems' => $data->pluck('id')
                ];
            }
            return ['data' => $data];
        }

        return view('admin.closure.index', ['data' => $data]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @throws AuthorizationException
     * @return Factory|View
     */
    public function create()
    {
        $this->authorize('admin.closure.create');

        return view('admin.closure.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreClosure $request
     * @return array|RedirectResponse|Redirector
     */
    public function store(StoreClosure $request)
    {
        // Sanitize input
        $sanitized = $request->getSanitized();

        // Store the Closure
        $closure = Closure::create($sanitized);

        if ($request->ajax()) {
            return ['redirect' => url('admin/closures'), 'message' => trans('brackets/admin-ui::admin.operation.succeeded')];
        }

        return redirect('admin/closures');
    }

    /**
     * Display the specified resource.
     *
     * @param Closure $closure
     * @throws AuthorizationException
     * @return void
     */
    public function show(Closure $closure)
    {
        $this->authorize('admin.closure.show', $closure);

        // TODO your code goes here
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Closure $closure
     * @throws AuthorizationException
     * @return Factory|View
     */
    public function edit(Closure $closure)
    {
        $this->authorize('admin.closure.edit', $closure);


        return view('admin.closure.edit', [
            'closure' => $closure,
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param UpdateClosure $request
     * @param Closure $closure
     * @return array|RedirectResponse|Redirector
     */
    public function update(UpdateClosure $request, Closure $closure)
    {
        // Sanitize input
        $sanitized = $request->getSanitized();

        // Update changed values Closure
        $closure->update($sanitized);

        if ($request->ajax()) {
            return [
                'redirect' => url('admin/closures'),
                'message' => trans('brackets/admin-ui::admin.operation.succeeded'),
            ];
        }

        return redirect('admin/closures');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param DestroyClosure $request
     * @param Closure $closure
     * @throws Exception
     * @return ResponseFactory|RedirectResponse|Response
     */
    public function destroy(DestroyClosure $request, Closure $closure)
    {
        $closure->delete();

        if ($request->ajax()) {
            return response(['message' => trans('brackets/admin-ui::admin.operation.succeeded')]);
        }

        return redirect()->back();
    }

    /**
     * Remove the specified resources from storage.
     *
     * @param BulkDestroyClosure $request
     * @throws Exception
     * @return Response|bool
     */
    public function bulkDestroy(BulkDestroyClosure $request) : Response
    {
        DB::transaction(static function () use ($request) {
            collect($request->data['ids'])
                ->chunk(1000)
                ->each(static function ($bulkChunk) {
                    Closure::whereIn('id', $bulkChunk)->delete();

                    // TODO your code goes here
                });
        });

        return response(['message' => trans('brackets/admin-ui::admin.operation.succeeded')]);
    }
}
